/*
 * MinIO Javascript Library for Amazon S3 Compatible Cloud Storage, (C) 2015, 2016 MinIO, Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

import * as Crypto from "crypto";
import Through2 from 'through2';
import { isFunction } from "./internal/helper.mjs";
import * as xmlParsers from "./xml-parsers.mjs";

// getConcater returns a stream that concatenates the input and emits
// the concatenated output when 'end' has reached. If an optional
// parser function is passed upon reaching the 'end' of the stream,
// `parser(concatenated_data)` will be emitted.
export function getConcater(parser, emitError) {
  var objectMode = false;
  var bufs = [];
  if (parser && !isFunction(parser)) {
    throw new TypeError('parser should be of type "function"');
  }
  if (parser) {
    objectMode = true;
  }
  return Through2({
    objectMode
  }, function (chunk, enc, cb) {
    bufs.push(chunk);
    cb();
  }, function (cb) {
    if (emitError) {
      cb(parser(Buffer.concat(bufs).toString()));
      // cb(e) would mean we have to emit 'end' by explicitly calling this.push(null)
      this.push(null);
      return;
    }
    if (bufs.length) {
      if (parser) {
        try {
          this.push(parser(Buffer.concat(bufs).toString()));
        } catch (e) {
          cb(e);
          return;
        }
      } else {
        this.push(Buffer.concat(bufs));
      }
    }
    cb();
  });
}

// A through stream that calculates md5sum and sha256sum
export function getHashSummer(enableSHA256) {
  var md5 = Crypto.createHash('md5');
  var sha256 = Crypto.createHash('sha256');
  return Through2.obj(function (chunk, enc, cb) {
    if (enableSHA256) {
      sha256.update(chunk);
    } else {
      md5.update(chunk);
    }
    cb();
  }, function (cb) {
    var md5sum = '';
    var sha256sum = '';
    if (enableSHA256) {
      sha256sum = sha256.digest('hex');
    } else {
      md5sum = md5.digest('base64');
    }
    var hashData = {
      md5sum,
      sha256sum
    };
    this.push(hashData);
    this.push(null);
    cb();
  });
}

// Following functions return a stream object that parses XML
// and emits suitable Javascript objects.

// Parses listObjects response.
export function getListObjectsV2Transformer() {
  return getConcater(xmlParsers.parseListObjectsV2);
}

// Parses listObjects with metadata response.
export function getListObjectsV2WithMetadataTransformer() {
  return getConcater(xmlParsers.parseListObjectsV2WithMetadata);
}

// Parses GET/SET BucketNotification response
export function getBucketNotificationTransformer() {
  return getConcater(xmlParsers.parseBucketNotification);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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