/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.services.util;

import jakarta.ws.rs.core.Response;
import jakarta.ws.rs.core.UriBuilder;
import jakarta.ws.rs.core.UriInfo;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.HashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.function.BooleanSupplier;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.codec.binary.Hex;
import org.keycloak.TokenVerifier;
import org.keycloak.common.Profile;
import org.keycloak.common.VerificationException;
import org.keycloak.common.util.Time;
import org.keycloak.crypto.KeyWrapper;
import org.keycloak.crypto.SignatureProvider;
import org.keycloak.crypto.SignatureVerifierContext;
import org.keycloak.events.EventBuilder;
import org.keycloak.exceptions.TokenVerificationException;
import org.keycloak.http.HttpRequest;
import org.keycloak.jose.jwk.JWK;
import org.keycloak.jose.jws.JWSHeader;
import org.keycloak.jose.jws.crypto.HashUtils;
import org.keycloak.models.ClientModel;
import org.keycloak.models.ClientSessionContext;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.ProtocolMapperModel;
import org.keycloak.models.RealmModel;
import org.keycloak.models.SingleUseObjectProvider;
import org.keycloak.models.UserSessionModel;
import org.keycloak.protocol.ProtocolMapper;
import org.keycloak.protocol.oidc.OIDCAdvancedConfigWrapper;
import org.keycloak.protocol.oidc.mappers.AbstractOIDCProtocolMapper;
import org.keycloak.protocol.oidc.mappers.OIDCAccessTokenMapper;
import org.keycloak.protocol.oidc.mappers.OIDCAccessTokenResponseMapper;
import org.keycloak.protocol.oidc.mappers.OIDCIDTokenMapper;
import org.keycloak.protocol.oidc.mappers.TokenIntrospectionTokenMapper;
import org.keycloak.protocol.oidc.mappers.UserInfoTokenMapper;
import org.keycloak.provider.ProviderConfigProperty;
import org.keycloak.representations.AccessToken;
import org.keycloak.representations.JsonWebToken;
import org.keycloak.representations.dpop.DPoP;
import org.keycloak.services.CorsErrorResponseException;
import org.keycloak.services.cors.Cors;
import org.keycloak.util.JWKSUtils;
import org.keycloak.utils.StringUtil;

public class DPoPUtil {
    public static final int DEFAULT_PROOF_LIFETIME = 10;
    public static final int DEFAULT_ALLOWED_CLOCK_SKEW = 15;
    public static final String DPOP_TOKEN_TYPE = "DPoP";
    public static final String DPOP_SCHEME = "DPoP";
    public static final String DPOP_SESSION_ATTRIBUTE = "dpop";
    public static final String DPOP_HTTP_HEADER = "DPoP";
    private static final String DPOP_JWT_HEADER_TYPE = "dpop+jwt";
    public static final String DPOP_ATH_ALG = "RS256";
    public static final Set<String> DPOP_SUPPORTED_ALGS = Stream.of("ES256", "ES384", "ES512", "PS256", "PS384", "PS512", "RS256", "RS384", "RS512").collect(Collectors.toSet());
    private static final Pattern WHITESPACES = Pattern.compile("\\s+");

    private static URI normalize(URI uri) {
        return UriBuilder.fromUri((URI)uri).replaceQuery("").build(new Object[0]);
    }

    public static Stream<Map.Entry<ProtocolMapperModel, ProtocolMapper>> getTransientProtocolMapper() {
        String PROVIDER_ID = "DPoP".toLowerCase(Locale.ROOT) + "-protocol-mapper";
        ProtocolMapperModel protocolMapperModel = new ProtocolMapperModel();
        protocolMapperModel.setId("DPoP");
        protocolMapperModel.setName("DPoP");
        protocolMapperModel.setProtocolMapper(PROVIDER_ID);
        protocolMapperModel.setProtocol("openid-connect");
        HashMap<String, String> config = new HashMap<String, String>();
        config.put("access.token.claim", "true");
        config.put("id.token.claim", "false");
        config.put("userinfo.token.claim", "false");
        config.put("introspection.token.claim", "false");
        protocolMapperModel.setConfig(config);
        DpopProtocolMapper dpopProtocolMapper = new DpopProtocolMapper(PROVIDER_ID);
        return Stream.of(Map.entry(protocolMapperModel, dpopProtocolMapper));
    }

    public static Optional<DPoP> retrieveDPoPHeaderIfPresent(KeycloakSession keycloakSession, OIDCAdvancedConfigWrapper clientConfig, EventBuilder event, Cors cors) {
        return DPoPUtil.retrieveDPoPHeaderIfPresent(keycloakSession, event, cors, () -> {
            HttpRequest request = keycloakSession.getContext().getHttpRequest();
            boolean isClientRequiresDpop = clientConfig != null && clientConfig.isUseDPoP();
            boolean isDpopHeaderPresent = request.getHttpHeaders().getHeaderString("DPoP") != null;
            return !isClientRequiresDpop && !isDpopHeaderPresent;
        });
    }

    public static Optional<DPoP> retrieveDPoPHeaderIfPresent(KeycloakSession keycloakSession, EventBuilder event, Cors cors) {
        return DPoPUtil.retrieveDPoPHeaderIfPresent(keycloakSession, event, cors, () -> {
            HttpRequest request = keycloakSession.getContext().getHttpRequest();
            boolean isDpopHeaderPresent = request.getHttpHeaders().getHeaderString("DPoP") != null;
            return !isDpopHeaderPresent;
        });
    }

    private static Optional<DPoP> retrieveDPoPHeaderIfPresent(KeycloakSession keycloakSession, EventBuilder event, Cors cors, BooleanSupplier isDPoPNotApplicableRequest) {
        if (!Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.DPOP) || isDPoPNotApplicableRequest.getAsBoolean()) {
            return Optional.empty();
        }
        HttpRequest request = keycloakSession.getContext().getHttpRequest();
        try {
            DPoP dPoP = new Validator(keycloakSession).request(request).uriInfo((UriInfo)keycloakSession.getContext().getUri()).validate();
            keycloakSession.setAttribute(DPOP_SESSION_ATTRIBUTE, (Object)dPoP);
            return Optional.of(dPoP);
        }
        catch (VerificationException ex) {
            event.detail("reason", ex.getMessage());
            event.error("invalid_dpop_proof");
            throw new CorsErrorResponseException(cors, "invalid_request", ex.getMessage(), Response.Status.BAD_REQUEST);
        }
    }

    private static DPoP validateDPoP(KeycloakSession session, URI uri, String method, String token, String accessToken, int lifetime, int clockSkew) throws VerificationException {
        JWSHeader header;
        if (token == null || token.trim().isEmpty()) {
            throw new VerificationException("DPoP proof is missing");
        }
        TokenVerifier verifier = TokenVerifier.create((String)token, DPoP.class);
        try {
            header = verifier.getHeader();
        }
        catch (VerificationException ex) {
            throw new VerificationException("DPoP header verification failure");
        }
        if (!DPOP_JWT_HEADER_TYPE.equals(header.getType())) {
            throw new VerificationException("Invalid or missing type in DPoP header: " + header.getType());
        }
        String algorithm = header.getAlgorithm().name();
        if (!DPOP_SUPPORTED_ALGS.contains(algorithm)) {
            throw new VerificationException("Unsupported DPoP algorithm: " + String.valueOf(header.getAlgorithm()));
        }
        JWK jwk = header.getKey();
        if (jwk == null) {
            throw new VerificationException("No JWK in DPoP header");
        }
        KeyWrapper key = JWKSUtils.getKeyWrapper((JWK)jwk);
        if (key.getPublicKey() == null) {
            throw new VerificationException("No public key in DPoP header");
        }
        if (key.getPrivateKey() != null) {
            throw new VerificationException("Private key is present in DPoP header");
        }
        key.setAlgorithm(header.getAlgorithm().name());
        SignatureVerifierContext signatureVerifier = ((SignatureProvider)session.getProvider(SignatureProvider.class, algorithm)).verifier(key);
        verifier.verifierContext(signatureVerifier);
        verifier.withChecks(new TokenVerifier.Predicate[]{DPoPClaimsCheck.INSTANCE, new DPoPHTTPCheck(uri, method), new DPoPIsActiveCheck(session, lifetime, clockSkew), new DPoPReplayCheck(session, lifetime)});
        if (accessToken != null) {
            verifier.withChecks(new TokenVerifier.Predicate[]{new DPoPAccessTokenHashCheck(accessToken)});
        }
        try {
            DPoP dPoP = (DPoP)verifier.verify().getToken();
            dPoP.setThumbprint(JWKSUtils.computeThumbprint((JWK)jwk));
            return dPoP;
        }
        catch (DPoPVerificationException ex) {
            throw ex;
        }
        catch (VerificationException ex) {
            throw new VerificationException("DPoP verification failure: " + ex.getMessage(), (Throwable)ex);
        }
    }

    public static TokenVerifier<AccessToken> withDPoPVerifier(TokenVerifier<AccessToken> verifier, RealmModel realm, Validator validator) {
        if (Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.DPOP)) {
            verifier = verifier.tokenType(List.of("Bearer", "DPoP")).withChecks(new TokenVerifier.Predicate[]{token -> {
                String[] split = WHITESPACES.split(validator.authHeader);
                String typeString = split[0];
                if (!typeString.equals("DPoP") && "DPoP".equals(token.getType())) {
                    throw new VerificationException("The access token type is DPoP but Authorization Header is not DPoP");
                }
                if (typeString.equals("DPoP") && !"DPoP".equals(token.getType())) {
                    throw new VerificationException("The access token type is not DPoP but Authorization Header is DPoP");
                }
                ClientModel clientModel = realm.getClientByClientId(token.getIssuedFor());
                if (clientModel == null) {
                    throw new VerificationException("Client not found");
                }
                if (OIDCAdvancedConfigWrapper.fromClientModel(clientModel).isUseDPoP() && !typeString.equals("DPoP")) {
                    throw new VerificationException("This client requires DPoP, but no DPoP Authorization header is present");
                }
                if (typeString.equals("DPoP")) {
                    if (validator.accessToken == null) {
                        throw new VerificationException("Access Token not set for validator");
                    }
                    DPoP dPoP = validator.validate();
                    DPoPUtil.validateBinding(token, dPoP);
                    if (!Objects.equals(token.getConfirmation().getKeyThumbprint(), dPoP.getThumbprint())) {
                        throw new VerificationException("DPoP Proof public key thumbprint does not match dpop_jkt");
                    }
                }
                return true;
            }});
        }
        return verifier;
    }

    public static void validateBinding(AccessToken token, DPoP dPoP) throws VerificationException {
        try {
            TokenVerifier.createWithoutSignature((JsonWebToken)token).withChecks(new TokenVerifier.Predicate[]{new DPoPBindingCheck(dPoP)}).verify();
        }
        catch (TokenVerificationException ex) {
            throw ex;
        }
        catch (VerificationException ex) {
            throw new VerificationException("Token verification failure", (Throwable)ex);
        }
    }

    public static void validateDPoPJkt(String dpopJkt, KeycloakSession session, EventBuilder event, Cors cors) {
        if (dpopJkt == null) {
            return;
        }
        DPoP dPoP = (DPoP)session.getAttribute(DPOP_SESSION_ATTRIBUTE, DPoP.class);
        if (dPoP == null) {
            String errorMessage = "DPoP Proof missing";
            event.detail("reason", errorMessage);
            event.error("invalid_request");
            throw new CorsErrorResponseException(cors, "invalid_request", errorMessage, Response.Status.BAD_REQUEST);
        }
        if (!dpopJkt.equals(dPoP.getThumbprint())) {
            String errorMessage = "DPoP Proof public key thumbprint does not match dpop_jkt";
            event.detail("reason", errorMessage);
            event.error("invalid_request");
            throw new CorsErrorResponseException(cors, "invalid_request", errorMessage, Response.Status.BAD_REQUEST);
        }
    }

    public static boolean isDPoPToken(AccessToken refreshToken) {
        return refreshToken.getConfirmation() != null && refreshToken.getConfirmation().getKeyThumbprint() != null;
    }

    private static final class DpopProtocolMapper
    extends AbstractOIDCProtocolMapper
    implements OIDCAccessTokenMapper,
    OIDCIDTokenMapper,
    UserInfoTokenMapper,
    TokenIntrospectionTokenMapper,
    OIDCAccessTokenResponseMapper {
        private final String providerId;

        public DpopProtocolMapper(String providerId) {
            this.providerId = providerId;
        }

        public String getDisplayCategory() {
            return "Token mapper";
        }

        public String getDisplayType() {
            return "DPoP";
        }

        public String getHelpText() {
            return "not needed";
        }

        public List<ProviderConfigProperty> getConfigProperties() {
            return List.of(new ProviderConfigProperty("multivalued", "multivalued", "", "boolean", (Object)false));
        }

        public String getId() {
            return this.providerId;
        }

        @Override
        public AccessToken transformAccessToken(AccessToken token, ProtocolMapperModel mappingModel, KeycloakSession session, UserSessionModel userSession, ClientSessionContext clientSessionCtx) {
            boolean isDPoPSupported = Profile.isFeatureEnabled((Profile.Feature)Profile.Feature.DPOP);
            if (!isDPoPSupported) {
                return super.transformAccessToken(token, mappingModel, session, userSession, clientSessionCtx);
            }
            DPoP dPoP = (DPoP)session.getAttribute(DPoPUtil.DPOP_SESSION_ATTRIBUTE, DPoP.class);
            if (dPoP == null) {
                return super.transformAccessToken(token, mappingModel, session, userSession, clientSessionCtx);
            }
            AccessToken.Confirmation confirmation = (AccessToken.Confirmation)token.getOtherClaims().get("cnf");
            if (confirmation == null) {
                confirmation = new AccessToken.Confirmation();
                token.setConfirmation(confirmation);
            }
            confirmation.setKeyThumbprint(dPoP.getThumbprint());
            token.type("DPoP");
            return super.transformAccessToken(token, mappingModel, session, userSession, clientSessionCtx);
        }
    }

    public static class Validator {
        private URI uri;
        private String method;
        private String dPoP;
        private String accessToken;
        private String authHeader;
        private int clockSkew = 15;
        private int lifetime = 10;
        private final KeycloakSession session;

        public Validator(KeycloakSession session) {
            this.session = session;
        }

        public Validator request(HttpRequest request) {
            this.uri = request.getUri().getAbsolutePath();
            this.method = request.getHttpMethod();
            this.dPoP = request.getHttpHeaders().getHeaderString("DPoP");
            this.authHeader = request.getHttpHeaders().getHeaderString("Authorization");
            return this;
        }

        public Validator dPoP(String dPoP) {
            this.dPoP = dPoP;
            return this;
        }

        public Validator accessToken(String accessToken) {
            this.accessToken = accessToken;
            return this;
        }

        public Validator uriInfo(UriInfo uriInfo) {
            this.uri = uriInfo.getAbsolutePath();
            return this;
        }

        public Validator uri(String uri) throws URISyntaxException {
            this.uri = new URI(uri);
            return this;
        }

        public Validator method(String method) {
            this.method = method;
            return this;
        }

        public DPoP validate() throws VerificationException {
            return DPoPUtil.validateDPoP(this.session, this.uri, this.method, this.dPoP, this.accessToken, this.lifetime, this.clockSkew);
        }
    }

    private static class DPoPClaimsCheck
    implements TokenVerifier.Predicate<DPoP> {
        static final TokenVerifier.Predicate<DPoP> INSTANCE = new DPoPClaimsCheck();

        private DPoPClaimsCheck() {
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            Long iat = t.getIat();
            String jti = t.getId();
            String htu = t.getHttpUri();
            String htm = t.getHttpMethod();
            if (iat != null && StringUtil.isNotBlank((String)jti) && StringUtil.isNotBlank((String)htm) && StringUtil.isNotBlank((String)htu)) {
                return true;
            }
            throw new DPoPVerificationException(t, "DPoP mandatory claims are missing");
        }
    }

    private static class DPoPHTTPCheck
    implements TokenVerifier.Predicate<DPoP> {
        private final URI uri;
        private final String method;

        DPoPHTTPCheck(URI uri, String method) {
            this.uri = uri;
            this.method = method;
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            try {
                if (!DPoPUtil.normalize(new URI(t.getHttpUri())).equals(DPoPUtil.normalize(this.uri))) {
                    throw new DPoPVerificationException(t, "DPoP HTTP URL mismatch");
                }
                if (!this.method.equals(t.getHttpMethod())) {
                    throw new DPoPVerificationException(t, "DPoP HTTP method mismatch");
                }
            }
            catch (URISyntaxException ex) {
                throw new DPoPVerificationException(t, "Malformed HTTP URL in DPoP proof");
            }
            return true;
        }
    }

    private static class DPoPIsActiveCheck
    implements TokenVerifier.Predicate<DPoP> {
        private final int lifetime;
        private final int clockSkew;

        public DPoPIsActiveCheck(KeycloakSession session, int lifetime, int clockSkew) {
            this.lifetime = lifetime;
            this.clockSkew = clockSkew;
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            long time = Time.currentTime();
            Long iat = t.getIat();
            if (iat > time + (long)this.clockSkew || iat < time - (long)this.lifetime - (long)this.clockSkew) {
                throw new DPoPVerificationException(t, "DPoP proof is not active");
            }
            return true;
        }
    }

    private static class DPoPReplayCheck
    implements TokenVerifier.Predicate<DPoP> {
        private final KeycloakSession session;
        private final int lifetime;

        public DPoPReplayCheck(KeycloakSession session, int lifetime) {
            this.session = session;
            this.lifetime = lifetime;
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            byte[] hash;
            String hashString;
            SingleUseObjectProvider singleUseCache = this.session.singleUseObjects();
            if (!singleUseCache.putIfAbsent(hashString = Hex.encodeHexString((byte[])(hash = HashUtils.hash((String)"SHA1", (byte[])(t.getId() + "\n" + t.getHttpUri()).getBytes()))), (long)((int)(t.getIat() + (long)this.lifetime - (long)Time.currentTime())))) {
                throw new DPoPVerificationException(t, "DPoP proof has already been used");
            }
            return true;
        }
    }

    private static class DPoPAccessTokenHashCheck
    implements TokenVerifier.Predicate<DPoP> {
        private final String hash;

        public DPoPAccessTokenHashCheck(String tokenString) {
            this.hash = HashUtils.accessTokenHash((String)DPoPUtil.DPOP_ATH_ALG, (String)tokenString, (boolean)true);
        }

        public boolean test(DPoP t) throws DPoPVerificationException {
            if (t.getAccessTokenHash() == null) {
                throw new DPoPVerificationException(t, "No access token hash in DPoP proof");
            }
            if (!t.getAccessTokenHash().equals(this.hash)) {
                throw new DPoPVerificationException(t, "DPoP proof access token hash mismatch");
            }
            return true;
        }
    }

    public static class DPoPVerificationException
    extends TokenVerificationException {
        public DPoPVerificationException(DPoP token, String message) {
            super((JsonWebToken)token, message);
        }
    }

    private static class DPoPBindingCheck
    implements TokenVerifier.Predicate<AccessToken> {
        private final DPoP proof;

        public DPoPBindingCheck(DPoP proof) {
            this.proof = proof;
        }

        public boolean test(AccessToken t) throws VerificationException {
            String thumbprint = this.proof.getThumbprint();
            AccessToken.Confirmation confirmation = t.getConfirmation();
            if (confirmation == null) {
                throw new TokenVerificationException((JsonWebToken)t, "No DPoP confirmation in access token");
            }
            String keyThumbprint = confirmation.getKeyThumbprint();
            if (keyThumbprint == null) {
                throw new TokenVerificationException((JsonWebToken)t, "No DPoP key thumbprint in access token");
            }
            if (!keyThumbprint.equals(thumbprint)) {
                throw new TokenVerificationException((JsonWebToken)t, "DPoP confirmation doesn't match DPoP proof");
            }
            return true;
        }
    }

    public static enum Mode {
        ENABLED,
        OPTIONAL,
        DISABLED;

    }
}

