from abc import ABC, abstractmethod

from typing import Optional

import markdown_it

from iotas.note import Note


class HtmlGenerator(ABC):
    """HTML generator base."""

    RESOURCE_CSS_PATH = "/media/css/web/markdown.css"
    RESOURCE_KATEX_CSS_PATH = "/media/css/web/katex.min.css"
    RESOURCE_KATEX_JS_PATH = "/media/js/katex.min.js"

    @abstractmethod
    def generate(
        self,
        note: Note,
        searching: bool,
        export_format: Optional[str],
        scroll_position: Optional[float] = None,
    ) -> tuple[str, list[markdown_it.token.Token]]:
        """Generator HTML for note.

        :param Note note: Note to render
        :param bool searching: Whether search CSS should be included
        :param bool export_format: Export format, if using
        :param Optional[float] scroll_position: Position to scroll to
        :return: Generated HTML and list of parser tokens
        :rtype: tuple[str, list[markdown_it.token.Token]]
        """
        raise NotImplementedError()

    @abstractmethod
    def generate_user_stylesheet(self, searching: bool) -> str:
        """Generate part of stylesheet based on state (configuration etc).

        :param bool searching: Whether searching
        :return: stylesheet
        :rtype: str
        """
        raise NotImplementedError()

    @abstractmethod
    def update_font_family(self, family: str) -> None:
        """Update the font family.

        :param str family: New font family
        """
        raise NotImplementedError()
